/* 
 *         GA Net - a genetic algorithm for generating Network Intusion Rules
 *
 *       Copyright (C) 2010 Brian E. Lavender <brian@brie.com> 
 *
 *                     http://www.brie.com/brian/ganet/ 
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <string.h>
#include <glib.h>
#include <glib/gprintf.h>
#include "types.h"
#include "print.h"
#include "rand.h"
#include "read_bsm.h"
#include "service_attacks.h"

#include "compare.h"

extern gint global_individual_count;

#ifndef SWAP_4
#define SWAP_4(x) ( ((x) << 24) | \
         (((x) << 8) & 0x00ff0000) | \
         (((x) >> 8) & 0x0000ff00) | \
         ((x) >> 24) )
#endif


int main() {
  GSList *auditList = NULL;
  GPtrArray  *popSrc, *popDest, *threeRand; // Array of individuals
  gdouble w1 = 0.2; // These constants are suggested per the paper
  gdouble w2 = 0.8;
  gdouble mutateProb = 0.05;
  gdouble wildCardProb = 0.05;
  guint nElite = 1000;
  guint nPop = 1000;
  guint i,j,k,m,n;
  GRand *rnd; // Random number entropy
  //  gdouble test_fit;

  GArray *myArrayL[NUM_HTABLES];
  GArray *myArrayC[NUM_HTABLES][SUBH];

  individual *trainer, *parent1, *parent2, *child1, *child2; // Random Individual
  guint nAuditRecords = 0; // number of audit records

  char *myfile = "./bsm.list";
  //char *myfile = "./bsm_pres1.list";

  g_mem_set_vtable(glib_mem_profiler_table);
  g_atexit(g_mem_profile);

  rnd = g_rand_new();

  g_print("Audit data pulled from %s\n",myfile);


  // Load the audit list
  nAuditRecords = build_audit_array(&auditList, myArrayL, myArrayC, myfile);

  // Build an array of training individuals now

  // Initialize the array
  popSrc = g_ptr_array_new_with_free_func(destroyInd);
  n=0;
  while (n < nPop) {
    makeRandIndV2(wildCardProb, &trainer, myArrayL, myArrayC);
    get_fitness(auditList, trainer, w1, w2, nAuditRecords); 
    //if ( trainer->fitness > 0.01 ) {
      g_ptr_array_add(popSrc, trainer); 
      n++;
      //} else
      //destroyInd(trainer);
  }

  g_ptr_array_sort(popSrc, sort_function);

  g_print("Initial population\n");
  for (i=0; i < popSrc->len; i++) {
    print_individual(g_ptr_array_index(popSrc, i));
    trainer = g_ptr_array_index(popSrc, i);
    g_print("fitness %.04f\n",trainer->fitness);
  }
  g_print("=========================================\n");


  // Start evolution process

  

  // m keeps track of the number of times through the cycle
  m = 0;
  do {

    popDest = g_ptr_array_new_with_free_func(destroyInd);

    // Sort source population
    g_ptr_array_sort(popSrc, sort_function);

    // Copy Elite individuals to popDest
    for (i=0, j= (popSrc->len) - nElite ; i<nElite ; i++,j++) { 
      trainer = g_ptr_array_index(popSrc, j);
      child1 = makeEmptyInd();
      g_memmove(child1,trainer, sizeof(individual) );
      g_ptr_array_add(popDest, child1 );
    }
    
    // Evolve for remainder of population
    for (k = 0; k < (nPop - nElite) /2 ; k++) {
      
      //threeRand = g_ptr_array_new_with_free_func(destroyInd);
      threeRand = g_ptr_array_new();
      for (i=0; i< 3 ;i++) {
	j = g_random_int_range(0, nPop);
	trainer = g_ptr_array_index(popSrc, j);
	g_ptr_array_add(threeRand, trainer);
      }

      g_ptr_array_sort(threeRand, sort_function);
    
      //g_print("Two top individuals are the following\n");

      // fitness goes lowest to highest (0,1,2). Thus elements 1 and 2
      parent1 = g_ptr_array_index(threeRand, 1);
      parent2 = g_ptr_array_index(threeRand, 2);

      g_ptr_array_free(threeRand, FALSE);

      child1 = makeEmptyInd();
      child2 = makeEmptyInd();

      // Uncomment for No breed. Just test
      //g_memmove(child1->chrome, parent1->chrome, NUM_GENE*4 );
      //g_memmove(child2->chrome, parent2->chrome, NUM_GENE*4 );

      breed_midpoint(parent1, parent2, child1, child2);
    
      mutateIndV1(rnd, mutateProb, wildCardProb, child1, myArrayL, myArrayC);
      mutateIndV1(rnd, mutateProb, wildCardProb, child2, myArrayL, myArrayC);
      
      get_fitness(auditList, child1, w1, w2, nAuditRecords); 
      get_fitness(auditList, child2, w1, w2, nAuditRecords); 

      g_ptr_array_add(popDest, child1);
      g_ptr_array_add(popDest, child2);

    }

    // Get fitness of the 10th highest
    //trainer = g_ptr_array_index(popDest, (popDest->len) - 10);
    //test_fit = trainer->fitness;

    // Shallow copy
    swapPop(&popDest,&popSrc);

    g_ptr_array_free(popDest, TRUE);
    m++;

  } while (m<30);

  g_ptr_array_sort(popSrc, sort_function);

  // Print the top 30 individuals
  g_print("Top 30 individuals are the following\n");
  for (i=0, j= (popSrc->len) - 30 ; i<30 ; i++,j++) { 
       trainer = g_ptr_array_index(popSrc, j);
       print_individual(trainer);       
       g_print("%d fitness is %.04f\n",j,trainer->fitness);
  }

  g_ptr_array_free(popSrc, TRUE);
  g_slist_free(auditList);
  g_rand_free(rnd);

  g_print("The net individuals should match the number of audit records %d.\n",
	  nAuditRecords);
  g_print("Count is %d.\n",global_individual_count);


  return 0;
  
}
